%==========================================================================
%
%  Diversity and Economic Performance in a Model with Progressive Taxation
%  
%  Authors: Wei Wang and Richard M.H. Suen
%  Date: 9th July, 2018
%
%  These codes are used to generate the main results reported in 
%  Section 4.3 of the paper.
%
%==========================================================================

close all;
clear all; 

global a0 a1 a2 a3 pi Income sigma g Output

pause on

%==========================================================================
%
% Part I: To obtain the baseline parameter values
%
%==========================================================================

alpha = 0.40;  % Capital's share of aggregate output
psi   = 0.40;  % Inverse of Frisch elasticity of labour supply

% Parameters in the progressive tax function
a0 = 0.258; 
a1 = 0.768; 

% Income Distribution from 2016 SCF
pi = [0.2 0.2 0.2 0.2 0.1 0.1];  % Population share of each group
s  = [1.0 2.0 3.2 5.5 8.2 28.5]; % Mean income relative to lowest group


M = length(pi);   % No. of income groups

% Labour Productivity (Uniform distribution)
epsi_min = 1;
epsi_max = 100;

% Targeted Statistics
K_Y_ratio    = 2.367;  % Capital-Output ratio
NI_GDP_ratio = 0.869;  % NNP-to-GDP ratio
g            = 0.207;  % Share of government spending in GDP
Hours        = 1/3;    % Average labour hours
sigma        = 0;      % Share of tax revenues from personal income tax


delta = (1 - NI_GDP_ratio)/K_Y_ratio; % Depreciation rate

a3 = g*sigma/NI_GDP_ratio; 

tau_bar = a0 + a3;

%============================================================================
% Steady State (as implied by the targeted statistics)
%============================================================================

k_ss = K_Y_ratio^(1/(1 - alpha));

r_ss = alpha*(k_ss^(alpha - 1)) - delta; 
w_ss = (1 - alpha)*(k_ss^alpha);

EX01 = (epsi_max^(2 + psi) - epsi_min^(2 + psi))/(epsi_max - epsi_min)/(2 + psi);
EX02 = (epsi_max^(1 + psi) - epsi_min^(1 + psi))/(epsi_max - epsi_min)/(1 + psi);

N_ss = EX01/EX02/3;

RHS    = ( k_ss^alpha - delta*k_ss )*N_ss; 

Aggregate_Capital = k_ss*N_ss;
Output = ( k_ss^alpha )*N_ss;  

Income = zeros(1,M);

Income(1) = RHS/(pi*s');
Income = Income(1)*s;

National_Income = pi*Income';

% Solve for a2 using the government's budget constraint

[sol, eval, flag] = fsolve('F01', 0, optimoptions('fsolve','Display','off')); 

if ( flag ~= 1 )
    
    fprintf('Error: fsolve is not converging.')
    %pause
    
end

a2 = sol;

tax  = @(x) a3*x + a0*( x - ( x.^(-a1) + a2 ).^(-1/a1) );
mtax = @(x) a3 + a0*( 1 - (1 + a2*x.^a1).^(-(1 + 1/a1)) );  

c_ss = Income - tax(Income);

rho = r_ss*(1 - mtax(Income) );   % Subjective discount factors

EX03 = (rho./c_ss).^psi;
EX04 = (pi*EX03').^(1/psi);

A = (w_ss/r_ss)*EX04*((EX02/Hours)^(1/psi));

labour = ( w_ss*rho/A/r_ss./c_ss ).^psi; 

%=======================================================================
% Checking

G_ss  = pi*( tax(Income)' );   % Level of government spending
G_ss2 = g*(k_ss^alpha)*N_ss;

check01 = G_ss/G_ss2 - 1;

if ( abs(check01) > 1e-05 )
    
    fprintf('Error in computing government spending.')
    %pause
    
end

max_labour = (w_ss*rho*epsi_max/r_ss/A./c_ss).^psi;

if ( max_labour > 1 )
    
    fprintf('Error: Maximum labour hours greater than one.')
    %pause
    
end

%=======================================================================

% Baseline Paramter values(Table 2)
[A psi alpha delta a0 a1 a2 a3 rho epsi_min epsi_max]

% Baseline Steady State (Table 3)

[k_ss N_ss National_Income Aggregate_Capital Output G_ss];


%========================================================================
%
%  Part II: Changing the Distribution of Subjective Discount Factors
%
%========================================================================

N_1 = 5;   % Number of Experiments

Results = zeros(N_1, M);

P = ones(N_1,1)*pi;

x = (rho(3) - rho(6))*0.05/(rho(1) - rho(6));

P(2,1) = pi(1) + x;
P(2,3) = pi(3) - 0.05;
P(2,6) = pi(6) + 0.05 - x;

x = (rho(3) - rho(6))*0.10/(rho(1) - rho(6));

P(3,1) = pi(1) + x;
P(3,3) = pi(3) - 0.10;
P(3,6) = pi(6) + 0.10 - x;

x = (rho(3) - rho(6))*0.20/(rho(1) - rho(6));

P(4,1) = pi(1) + x;
P(4,3) = pi(3) - 0.20;
P(4,6) = pi(6) + 0.20 - x;

x = (rho(3) - rho(6))*(-0.10)/(rho(1) - rho(6));

P(5,1) = pi(1) + x;
P(5,3) = pi(3) + 0.10;
P(5,6) = pi(6) - 0.10 - x;

%===========================================================================

rho_max = max(rho);
rho_min = min(rho);

k_max = ( alpha/(rho_max + delta) )^(1/(1 - alpha));
k_min = ( alpha*(1 - tau_bar)/( rho_min + (1 - tau_bar)*delta ) )^(1/(1 - alpha));

if ( k_min >= k_max )
    
     fprintf('Error: The range of plausible k is empty!')
     pause
     
end

Tol = 1e-06;

labour2 = zeros(N_1, M);
Income2 = zeros(N_1, M);

for i = 1:N_1
    
    i;
    
    k2_min = 1.00001*k_min;
    k2_max = 0.995*k_max;
    
    k   = 0;
    Ind = 0;
    Error = -999;
    
      
    while ( abs(Error) > Tol )
        
        k = ( k2_min + k2_max )/2;
        
        r = alpha*(k^(alpha - 1)) - delta;
        w = (1 - alpha)*(k^alpha);
        
        F01   = @(x) 1 - (1 - a3 - x/r)/a0;
        F02   = @(x) ( F01(x) ).^(-a1/(1 + a1)) - 1;
        y_fun = @(x) ( F02(x)/a2 ).^(1/a1);
        
        Income3 = y_fun(rho);
        
        National_Income2 = P(i,:)*Income3';
        
        c2 = Income3 - tax(Income3);
        
        EX10 = (w/A/r)^psi;
        EX11 = P(i,:)*( ( rho./c2 ).^psi )';
        
        Aggregate_Labour = EX10*EX11*EX01;
        
        RHS = ( k^alpha - delta*k )*Aggregate_Labour;
        
        Error = National_Income2/RHS - 1;
            
        %===================================================================
        if ( Error > 0 )
            
            k2_min = k;
            
        else
            
            k2_max = k;
            
        end
        %====================================================================
        
        Ind = Ind + 1;
        
        if (Ind > 100 )
            
            fprintf('While Loop does not converge!')
            pause
            
        end
        
    end;
    
    
    r = alpha*(k^(alpha - 1)) - delta;
    w = (1 - alpha)*(k^alpha);
    
    F01   = @(x) 1 - (1 - a3 - x/r)/a0;
    F02   = @(x) ( F01(x) ).^(-a1/(1 + a1)) - 1;
    y_fun = @(x) ( F02(x)/a2 ).^(1/a1);
    
    Income2(i,:) = y_fun(rho);
    
    National_Income2 = P(i,:)*Income2(i,:)';
    
    c2 = Income2(i,:) - tax(Income2(i,:));
    
    labour2(i,:) = ( w*rho/A/r./c2 ).^psi;
    
    %=================================================================
    % Checking
    
    max_labour2 = (w*rho*epsi_max/r/A./c2).^psi;
    
    if ( max_labour2 > 1 )
        
        fprintf('Error: Maximum labour hours greater than one.')
        pause
        
    end
    
    %==================================================================
    
    EX10 = (w/A/r)^psi;
    EX11 = P(i,:)*( ( rho./c2 ).^psi )';
    
    Aggregate_Labour2 = EX10*EX11*EX01;
    
    Output2 = ( k^alpha )*Aggregate_Labour2;
    
    Aggregate_Capital2 = k*Aggregate_Labour2;
    
    G2 = P(i,:)*( tax(Income2(i,:))' );
    
    Results(i,:) = [k Aggregate_Labour2 National_Income2 Aggregate_Capital2 Output2 G2];
    
end

Results

% Table 5: Composition Effects vs G.E. Effects

Results2 = (( EX01*P*labour2(1,:)' )/( EX01*P(1,:)*labour2(1,:)' ) - 1)*100

Results3 = (( EX01*P(1,:)*labour2' )/( EX01*P(1,:)*labour2(1,:)' ) - 1)*100

Results4 = (( P*Income2(1,:)' )/( P(1,:)*Income2(1,:)' ) - 1)*100

Results5 = (( P(1,:)*Income2' )/( P(1,:)*Income2(1,:)' ) - 1)*100


%=========================================================================
%
%  Part III: Drawing the diagrams
%
%=========================================================================

k_up = 1.02*k_ss; 
k_low = max( 0.98*k_ss, 1.0001*k_min);

N_2 = 500; % Number of points in capital grid

k_vec = linspace(k_low, k_up, N_2);

r_vec = alpha*(k_vec.^(alpha - 1)) - delta;
w_vec = (1 - alpha)*(k_vec.^alpha);

EX20 = (w_vec/A./r_vec).^psi;
    
y = zeros(M, N_2);
n = zeros(M, N_2);

Agg_Labour = zeros(N_1, N_2);
Nat_Income = zeros(N_1, N_2);

for i = 1:N_1
    
    for j = 1:M
              
        F01 = 1 - (1 - a3 - rho(j)./r_vec)/a0;
        F02 = F01.^(-a1/(1 + a1)) - 1;
        y(j,:) = ( F02/a2 ).^(1/a1);
        
        c = y(j,:) - tax(y(j,:));
                      
        n(j,:) = (rho(j)./c).^psi;
        
    end
    
    
    Agg_Labour(i,:) = EX01*EX20.*(P(i,:)*n);
    
    Nat_Income(i,:) = P(i,:)*y;
     
end


subplot(1,2,1)
plot(k_vec, Nat_Income(1,:));
hold on
%plot(k_vec, Nat_Income(2,:), ':');
plot(k_vec, Nat_Income(3,:), '--');
plot(k_vec, Nat_Income(4,:), '-.');
plot(k_vec, Nat_Income(5,:), ':' );
hold off

subplot(1,2,2)
plot(k_vec, Agg_Labour(1,:));
hold on
%plot(k_vec, Agg_Labour(2,:), ':');
plot(k_vec, Agg_Labour(3,:), '--');
plot(k_vec, Agg_Labour(4,:), '-.');
plot(k_vec, Agg_Labour(5,:), ':');
hold off








